<?php

namespace App\Http\Controllers\Api;

use App\Helpers\AppHelper;
use App\Helpers\AttendanceHelper;
use App\Helpers\NepaliDate;
use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\User;
use App\Repositories\EmployeePayslipAdditionalRepository;
use App\Requests\Payroll\Payslip\PayslipRequest;
use App\Resources\Payroll\Payslip\PayrollCollection;
use App\Resources\Payroll\Payslip\PayrollResource;
use App\Resources\Payroll\Payslip\PayslipResource;
use App\Services\Payroll\GeneratePayrollService;
use App\Services\Payroll\UnderTimeSettingService;
use App\Traits\CustomAuthorizesRequests;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\View;
use MilanTarami\NumberToWordsConverter\Services\NumberToWords;

class EmployeePayrollApiController extends Controller
{
    use CustomAuthorizesRequests;
    public function __construct(protected GeneratePayrollService $generatePayrollService, protected UnderTimeSettingService $utSettingService
    ,protected EmployeePayslipAdditionalRepository $additionalRepository){}

    public function getPayrollList(PayslipRequest $request): JsonResponse
    {
        $this->authorize('view_payslip_list');

        try {


            $validatedData = $request->validated();
            $isBsEnabled = AppHelper::ifDateInBsEnabled();

            $userId = getAuthUserCode();

            if(!empty($validatedData['year'])){
                if($isBsEnabled)
                {

                    $dateInAD = AppHelper::findAdDatesFromNepaliMonthAndYear($validatedData['year'], $validatedData['month']);

                    $startDate = date('Y-m-d',strtotime($dateInAD['start_date'])) ?? null;
                    $endDate = date('Y-m-d',strtotime($dateInAD['end_date'])) ?? null;
                }else{
                    $firstDayOfMonth  = Carbon::create($validatedData['year'], $validatedData['month'], 1)->startOfDay();
                    $startDate = date('Y-m-d',strtotime($firstDayOfMonth));
                    $endDate = date('Y-m-d',strtotime($firstDayOfMonth->endOfMonth()));
                }
                $payslip = $this->generatePayrollService->getEmployeePayslip($userId, $startDate, $endDate, $isBsEnabled);
                $payslipData = new PayrollCollection($payslip);
            }else{

                $payslip = $this->generatePayrollService->getPaidEmployeePayslip($userId,$isBsEnabled);

                $payslipData = new PayrollCollection($payslip);
            }



            $currency = AppHelper::getCompanyPaymentCurrencySymbol();



            $data =[
                'payslip'=>$payslipData,
                'currency'=>$currency
            ];
            return AppHelper::sendSuccessResponse(__('index.data_found'),$data);
        } catch (Exception $exception) {
            return AppHelper::sendErrorResponse($exception->getMessage(), $exception->getCode());
        }
    }

    public function getEmployeePayslipDetailById($id)
    {
        try {
            $this->authorize('view_payslip_detail');

            $payrollData = $this->generatePayrollService->getEmployeePayslipDetail($id);

            $components = $this->generatePayrollService->getEmployeePayslipDetailData($id);
            $earnings = array_values(array_filter($components, function ($component) {
                return $component['component_type'] == 'earning';
            }));

            $deductions = array_values(array_filter($components, function ($component) {
                return $component['component_type'] == 'deductions';
            }));

            $additionalData = $this->additionalRepository->getAll($id);
            $additionalComponent= $additionalData->toArray();
            $additionalEarnings = array_values(array_filter($additionalComponent, function ($component) {
                return $component['component_type'] == 'earning';
            }));

            $additionalDeductions = array_values(array_filter($additionalComponent, function ($component) {
                return $component['component_type'] == 'deductions';
            }));




            /** for pdf view */
            $companyLogoPath = Company::UPLOAD_PATH;
            $currency = AppHelper::getCompanyPaymentCurrencySymbol();
            $underTimeSetting = $this->utSettingService->getAllUTList(['is_active'],1);
            $numberToWords = new NumberToWords();

            $html = View::make('admin.payroll.employeeSalary.download_payslip', compact('payrollData','additionalEarnings','additionalDeductions','earnings','deductions', 'underTimeSetting','numberToWords', 'companyLogoPath','currency'))->render();

            $mergeEarning = array_merge($earnings,$additionalEarnings);
            $mergeDeductions = array_merge($deductions,$additionalDeductions);

            /** resource for payslip data */
            $payslipDetailData = new PayslipResource($payrollData);

            $data =[
                'payslipData'=>$payslipDetailData,
                'currency'=>$currency,
                'earnings'=>$mergeEarning,
                'deductions'=>$mergeDeductions,
                'file'=>$html,
            ];


            return AppHelper::sendSuccessResponse(__('index.data_found'),$data);
        } catch (Exception $exception) {
            return AppHelper::sendErrorResponse($exception->getMessage(), $exception->getCode());
        }
    }

    public function ssfHistory(Request $request)
    {
        try {

            $employeeId = getAuthUserCode();
            if ($request->all()) {
                $filterParameter = [
                    'year' => $request->year ?? null,
                ];


                $firstDayOfMonth = Carbon::create($filterParameter['year'], 01, 01)->startOfDay();
                $startDate = date('Y-m-d', strtotime($firstDayOfMonth));
                $lastDayOfMonth = Carbon::create($filterParameter['year'], 12, 01)->startOfDay();
                $endDate = date('Y-m-d', strtotime($lastDayOfMonth->endOfMonth()));

                $ssfHistory = $this->generatePayrollService->getEmployeeSsfHistory($employeeId, $startDate, $endDate);
            } else {
                $ssfHistory = $this->generatePayrollService->getRecentEmployeeSsf($employeeId);
            }

            $currency = AppHelper::getCompanyPaymentCurrencySymbol();
            $data =[
                'currency'=>$currency,
            ];
            return AppHelper::sendSuccessResponse(__('index.data_found'),$data);

        } catch (Exception $exception) {
            return AppHelper::sendErrorResponse($exception->getMessage(), $exception->getCode());
        }
    }

}
